<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/api/esim-api.php';

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: /login.php');
    exit;
}

// Get eSIM ID
$esimId = isset($_GET['id']) ? intval($_GET['id']) : 0;

if (!$esimId) {
    header('Location: /my-esims.php');
    exit;
}

// Get eSIM from database
$conn = getDBConnection();
$stmt = $conn->prepare("SELECT * FROM orders WHERE id = ? AND user_id = ?");
$stmt->bind_param("ii", $esimId, $_SESSION['user_id']);
$stmt->execute();
$result = $stmt->get_result();
$esim = $result->fetch_assoc();
$stmt->close();
$conn->close();

if (!$esim) {
    header('Location: /my-esims.php');
    exit;
}

// Initialize API
$api = new eSIMAPI();

// Initialize variables
$apiData = null;
$isExpired = false;
$isActive = false;
$totalData = null;
$usedData = null;
$remainingData = null;
$expiryDate = null;
$activationTime = null;
$network = null;

// Try to get API data
if (!empty($esim['iccid'])) {
    try {
        $apiResponse = $api->queryProfile($esim['iccid']);
        
        $isSuccess = false;
        if (isset($apiResponse['code']) && $apiResponse['code'] == 0) {
            $isSuccess = true;
        } elseif (isset($apiResponse['success']) && $apiResponse['success'] === true) {
            $isSuccess = true;
        } elseif (isset($apiResponse['errorCode']) && $apiResponse['errorCode'] === '0') {
            $isSuccess = true;
        }
        
        if ($apiResponse && $isSuccess) {
            $esimList = $apiResponse['obj']['esimList'] ?? ($apiResponse['obj'] ?? null);
            
            if (is_array($esimList) && !empty($esimList)) {
                $apiData = is_array($esimList[0]) ? $esimList[0] : $esimList;
                
                // وضعیت: esimStatus (ACTIVE, CANCEL, etc.)
                $isActive = isset($apiData['esimStatus']) && $apiData['esimStatus'] == 'ACTIVE';
                
                // تاریخ انقضا: expiredTime
                if (isset($apiData['expiredTime'])) {
                    $expiryDate = $apiData['expiredTime'];
                    $expiryTimestamp = strtotime($expiryDate);
                    $isExpired = $expiryTimestamp < time();
                }
                
                // زمان فعال‌سازی: activateTime
                if (isset($apiData['activateTime']) && $apiData['activateTime']) {
                    $activationTime = $apiData['activateTime'];
                }
                
                // APN (به عنوان اپراتور/شبکه): apn
                if (isset($apiData['apn']) && $apiData['apn']) {
                    $network = $apiData['apn'];
                }
                
                // حجم داده: totalVolume و orderUsage
                if (isset($apiData['totalVolume'])) {
                    $totalData = $apiData['totalVolume']; // به بایت
                    $usedData = $apiData['orderUsage'] ?? 0; // به بایت
                    $remainingData = $totalData - $usedData;
                }
            }
        }
    } catch (Exception $e) {
        // Silently fail
    }
}

// Fallback to local data
if (!$apiData && isset($esim['validity_days']) && isset($esim['created_at'])) {
    $createdTime = strtotime($esim['created_at']);
    $expiryTime = $createdTime + ($esim['validity_days'] * 24 * 60 * 60);
    $isExpired = $expiryTime < time();
    $expiryDate = date('Y-m-d H:i:s', $expiryTime);
}

$usagePercentage = 0;
if ($totalData > 0 && $usedData !== null) {
    $usagePercentage = round(($usedData / $totalData) * 100, 1);
}

$statusText = $isExpired ? 'منقضی' : 'فعال';
$statusIcon = $isExpired ? 'fa-times-circle' : 'fa-check-circle';
$statusClass = $isExpired ? 'status-expired' : 'status-active';

// Days remaining
$daysRemaining = 0;
if ($expiryDate) {
    $expiryTimestamp = strtotime($expiryDate);
    $daysRemaining = max(0, floor(($expiryTimestamp - time()) / (60 * 60 * 24)));
} elseif (isset($esim['validity_days']) && isset($esim['created_at'])) {
    $createdTime = strtotime($esim['created_at']);
    $expiryTime = $createdTime + ($esim['validity_days'] * 24 * 60 * 60);
    $daysRemaining = max(0, floor(($expiryTime - time()) / (60 * 60 * 24)));
}

include __DIR__ . '/includes/header.php';
?>

<link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@400;500;600;700;800&display=swap" rel="stylesheet">

<style>
* {
    font-family: 'Vazirmatn', Tahoma, Arial, sans-serif;
}

body {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    min-height: 100vh;
    padding: 20px;
    direction: rtl;
}

.back-button {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    background: white;
    color: #6b7ff5;
    padding: 10px 20px;
    border-radius: 10px;
    text-decoration: none;
    font-weight: 600;
    margin-bottom: 20px;
    transition: all 0.3s ease;
}

.back-button:hover {
    background: #6b7ff5;
    color: white;
    transform: translateX(-5px);
}

.details-container {
    max-width: 900px;
    margin: 0 auto;
}

.esim-header {
    background: white;
    border-radius: 20px;
    padding: 30px;
    margin-bottom: 20px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.2);
    text-align: center;
}

.header-flag {
    width: 120px;
    height: 85px;
    margin: 0 auto 20px;
    border-radius: 15px;
    overflow: hidden;
    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
}

.header-flag img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.header-country {
    font-size: 2rem;
    font-weight: 800;
    color: #1e293b;
    margin-bottom: 10px;
}

.header-package {
    font-size: 1.2rem;
    color: #64748b;
    font-weight: 600;
    margin-bottom: 15px;
}

.status-badge-large {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 10px 20px;
    border-radius: 25px;
    font-weight: 700;
    font-size: 1rem;
}

.status-active {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.status-expired {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.details-grid {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 20px;
    margin-bottom: 20px;
}

.detail-card {
    background: white;
    border-radius: 15px;
    padding: 25px;
    box-shadow: 0 5px 20px rgba(0,0,0,0.15);
}

.detail-card-title {
    display: flex;
    align-items: center;
    gap: 10px;
    font-size: 1.1rem;
    font-weight: 700;
    color: #475569;
    margin-bottom: 20px;
}

.detail-card-title i {
    color: #6b7ff5;
    font-size: 1.3rem;
}

.full-width {
    grid-column: 1 / -1;
}

/* Circular Progress */
.circular-progress-wrapper {
    display: flex;
    align-items: center;
    justify-content: space-around;
    gap: 30px;
}

.circular-progress {
    position: relative;
    width: 180px;
    height: 180px;
    flex-shrink: 0;
}

.circular-progress-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    text-align: center;
}

.circular-percentage {
    font-size: 2.5rem;
    font-weight: 800;
    color: #6b7ff5;
    line-height: 1;
    margin-bottom: 5px;
}

.circular-label {
    font-size: 0.8rem;
    color: #64748b;
    font-weight: 600;
}

.circular-stats {
    display: flex;
    flex-direction: column;
    gap: 15px;
    flex: 1;
}

.circular-stat-item {
    display: flex;
    align-items: center;
    gap: 12px;
    background: #f8fafc;
    padding: 12px;
    border-radius: 12px;
}

.circular-stat-icon {
    width: 45px;
    height: 45px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
    flex-shrink: 0;
}

.circular-stat-content {
    flex: 1;
}

.circular-stat-label {
    font-size: 0.75rem;
    color: #64748b;
    font-weight: 600;
    margin-bottom: 3px;
}

.circular-stat-value {
    font-size: 1rem;
    color: #1e293b;
    font-weight: 700;
}

.not-activated-message {
    display: flex;
    align-items: center;
    gap: 15px;
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    padding: 20px;
    border-radius: 12px;
    color: white;
}

.not-activated-icon {
    width: 50px;
    height: 50px;
    background: rgba(255,255,255,0.25);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    flex-shrink: 0;
}

.not-activated-title {
    font-size: 1rem;
    font-weight: 700;
    margin-bottom: 5px;
}

.not-activated-desc {
    font-size: 0.85rem;
    opacity: 0.95;
}

/* Time Box */
.time-box {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border-radius: 15px;
    padding: 25px;
    display: flex;
    align-items: center;
    gap: 20px;
    color: white;
}

.time-icon-large {
    width: 70px;
    height: 70px;
    background: rgba(255,255,255,0.25);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 2rem;
    flex-shrink: 0;
}

.time-content {
    flex: 1;
}

.time-label {
    font-size: 0.9rem;
    opacity: 0.9;
    margin-bottom: 5px;
}

.time-value {
    font-size: 2.5rem;
    font-weight: 800;
}

/* Info List */
.info-list {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.info-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 0;
    border-bottom: 1px solid #e2e8f0;
}

.info-row:last-child {
    border-bottom: none;
}

.info-label {
    color: #64748b;
    font-weight: 600;
    font-size: 0.9rem;
}

.info-value {
    color: #1e293b;
    font-weight: 700;
    font-size: 0.95rem;
}

.status-active-text {
    color: #10b981;
}

.status-inactive-text {
    color: #ef4444;
}

/* QR Code */
.qr-wrapper {
    text-align: center;
    margin-bottom: 20px;
}

.qr-wrapper img {
    max-width: 250px;
    border-radius: 15px;
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
}

.btn-auto-add {
    display: inline-flex;
    align-items: center;
    gap: 10px;
    background: linear-gradient(135deg, #6b7ff5 0%, #8a96f7 100%);
    color: white;
    padding: 12px 25px;
    border-radius: 12px;
    text-decoration: none;
    font-weight: 700;
    margin-top: 15px;
    transition: all 0.3s ease;
}

.btn-auto-add:hover {
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(107, 127, 245, 0.4);
}

/* Activation Code */
.activation-code-box {
    background: #f8fafc;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    padding: 15px;
    margin-bottom: 15px;
}

.activation-label {
    color: #64748b;
    font-weight: 600;
    font-size: 0.85rem;
    margin-bottom: 8px;
    display: block;
}

.activation-code {
    font-family: 'Courier New', monospace;
    background: white;
    padding: 12px;
    border-radius: 8px;
    border: 1px solid #e2e8f0;
    font-size: 0.85rem;
    color: #1e293b;
    word-break: break-all;
    margin-bottom: 10px;
}

.btn-copy {
    background: #6b7ff5;
    color: white;
    border: none;
    padding: 10px 20px;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 8px;
    width: 100%;
    justify-content: center;
    transition: all 0.3s ease;
}

.btn-copy:hover {
    background: #5a6de8;
}

/* Instructions */
.instructions-content {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.instruction-section {
    background: #f8fafc;
    padding: 20px;
    border-radius: 12px;
    border: 1px solid #e2e8f0;
}

.instruction-section h4 {
    color: #1e293b;
    font-size: 1.1rem;
    font-weight: 700;
    margin-bottom: 15px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.instruction-section h4 i {
    color: #6b7ff5;
    font-size: 1.3rem;
}

.instruction-section ol {
    margin: 0;
    padding-right: 20px;
    color: #475569;
}

.instruction-section li {
    margin-bottom: 10px;
    line-height: 1.6;
}

.instruction-section li:last-child {
    margin-bottom: 0;
}

.note-box {
    background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
    border: 2px solid #fbbf24;
    border-radius: 12px;
    padding: 15px;
    display: flex;
    align-items: flex-start;
    gap: 12px;
}

.note-box i {
    color: #d97706;
    font-size: 1.3rem;
    flex-shrink: 0;
    margin-top: 2px;
}

.note-box strong {
    color: #92400e;
}

@media (max-width: 768px) {
    .details-grid {
        grid-template-columns: 1fr;
    }
    
    .circular-progress-wrapper {
        flex-direction: column;
    }
    
    .time-box {
        flex-direction: column;
        text-align: center;
    }
}
</style>

<div class="details-container">
    <a href="/my-esims.php" class="back-button">
        <i class="fas fa-arrow-right"></i>
        <span>بازگشت به لیست</span>
    </a>
    
    <!-- Header -->
    <div class="esim-header">
        <div class="header-flag">
            <?php if (!empty($esim['country_code'])): ?>
            <img src="https://flagcdn.com/w320/<?php echo strtolower($esim['country_code']); ?>.png" 
                 alt="<?php echo htmlspecialchars($esim['country_name'] ?? ''); ?>">
            <?php endif; ?>
        </div>
        
        <div class="header-country"><?php echo htmlspecialchars($esim['country_name'] ?? ''); ?></div>
        <div class="header-package"><?php echo htmlspecialchars($esim['package_name'] ?? 'eSIM'); ?></div>
        
        <div class="status-badge-large <?php echo $statusClass; ?>">
            <i class="fas <?php echo $statusIcon; ?>"></i>
            <span><?php echo $statusText; ?></span>
        </div>
    </div>
    
    <!-- Details Grid -->
    <div class="details-grid">
        <!-- Data Usage -->
        <div class="detail-card full-width">
            <div class="detail-card-title">
                <i class="fas fa-chart-pie"></i>
                <span>مصرف حجم داده</span>
            </div>
            
            <?php if ($totalData !== null && $usedData !== null): ?>
            <div class="circular-progress-wrapper">
                <div class="circular-progress">
                    <svg width="180" height="180" viewBox="0 0 180 180">
                        <circle cx="90" cy="90" r="75" fill="none" stroke="#e2e8f0" stroke-width="15"/>
                        <circle cx="90" cy="90" r="75" fill="none" 
                                stroke="url(#gradient)" 
                                stroke-width="15"
                                stroke-dasharray="<?php echo ($usagePercentage * 471.24 / 100); ?> 471.24"
                                stroke-linecap="round"
                                transform="rotate(-90 90 90)"/>
                        <defs>
                            <linearGradient id="gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                                <stop offset="0%" style="stop-color:#6b7ff5;stop-opacity:1" />
                                <stop offset="100%" style="stop-color:#8a96f7;stop-opacity:1" />
                            </linearGradient>
                        </defs>
                    </svg>
                    
                    <div class="circular-progress-text">
                        <div class="circular-percentage"><?php echo $usagePercentage; ?>%</div>
                        <div class="circular-label">مصرف شده</div>
                    </div>
                </div>
                
                <div class="circular-stats">
                    <div class="circular-stat-item">
                        <div class="circular-stat-icon" style="background: linear-gradient(135deg, #6b7ff5 0%, #8a96f7 100%);">
                            <i class="fas fa-arrow-up"></i>
                        </div>
                        <div class="circular-stat-content">
                            <div class="circular-stat-label">مصرف شده</div>
                            <div class="circular-stat-value"><?php echo round($usedData / 1024 / 1024 / 1024, 2); ?> GB</div>
                        </div>
                    </div>
                    
                    <div class="circular-stat-item">
                        <div class="circular-stat-icon" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                            <i class="fas fa-arrow-down"></i>
                        </div>
                        <div class="circular-stat-content">
                            <div class="circular-stat-label">باقیمانده</div>
                            <div class="circular-stat-value"><?php echo round($remainingData / 1024 / 1024 / 1024, 2); ?> GB</div>
                        </div>
                    </div>
                    
                    <div class="circular-stat-item">
                        <div class="circular-stat-icon" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                            <i class="fas fa-database"></i>
                        </div>
                        <div class="circular-stat-content">
                            <div class="circular-stat-label">کل حجم</div>
                            <div class="circular-stat-value"><?php echo round($totalData / 1024 / 1024 / 1024, 2); ?> GB</div>
                        </div>
                    </div>
                </div>
            </div>
            <?php else: ?>
            <div class="not-activated-message">
                <div class="not-activated-icon">
                    <i class="fas fa-info-circle"></i>
                </div>
                <div>
                    <div class="not-activated-title">eSIM هنوز فعال نشده است</div>
                    <div class="not-activated-desc">پس از فعال‌سازی eSIM، اطلاعات مصرف داده در اینجا نمایش داده می‌شود.</div>
                </div>
            </div>
            <?php endif; ?>
        </div>
        
        <!-- Time Remaining -->
        <div class="detail-card">
            <div class="time-box">
                <div class="time-icon-large">
                    <i class="fas fa-hourglass-half"></i>
                </div>
                <div class="time-content">
                    <div class="time-label">مدت زمان باقیمانده</div>
                    <div class="time-value"><?php echo $daysRemaining; ?> روز</div>
                </div>
            </div>
        </div>
        
        <!-- Network Info -->
        <div class="detail-card">
            <div class="detail-card-title">
                <i class="fas fa-broadcast-tower"></i>
                <span>اطلاعات شبکه</span>
            </div>
            
            <div class="info-list">
                <?php if ($apiData && isset($apiData['esimStatus'])): ?>
                <div class="info-row">
                    <span class="info-label">وضعیت eSIM:</span>
                    <span class="info-value <?php echo $isActive ? 'status-active-text' : 'status-inactive-text'; ?>">
                        <?php 
                        $statusMap = [
                            'ACTIVE' => 'فعال',
                            'CANCEL' => 'لغو شده',
                            'RELEASED' => 'آماده نصب',
                            'INSTALLED' => 'نصب شده'
                        ];
                        echo $statusMap[$apiData['esimStatus']] ?? htmlspecialchars($apiData['esimStatus']);
                        ?>
                    </span>
                </div>
                <?php endif; ?>
                
                <?php if ($network): ?>
                <div class="info-row">
                    <span class="info-label">APN / شبکه:</span>
                    <span class="info-value"><?php echo htmlspecialchars($network); ?></span>
                </div>
                <?php endif; ?>
                
                <?php if ($apiData && isset($apiData['ipExport'])): ?>
                <div class="info-row">
                    <span class="info-label">IP Export:</span>
                    <span class="info-value"><?php echo htmlspecialchars($apiData['ipExport']); ?></span>
                </div>
                <?php endif; ?>
                
                <div class="info-row">
                    <span class="info-label">حجم کل:</span>
                    <span class="info-value"><?php echo htmlspecialchars($esim['data_amount'] ?? 'N/A'); ?></span>
                </div>
                
                <?php if ($apiData && isset($apiData['totalDuration'])): ?>
                <div class="info-row">
                    <span class="info-label">مدت اعتبار:</span>
                    <span class="info-value"><?php echo $apiData['totalDuration']; ?> روز</span>
                </div>
                <?php endif; ?>
                
                <?php if ($activationTime): ?>
                <div class="info-row">
                    <span class="info-label">زمان فعال‌سازی:</span>
                    <span class="info-value"><?php echo date('Y/m/d H:i', strtotime($activationTime)); ?></span>
                </div>
                <?php else: ?>
                <div class="info-row">
                    <span class="info-label">زمان فعال‌سازی:</span>
                    <span class="info-value" style="color: #f59e0b;">هنوز فعال نشده</span>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- QR Code -->
        <?php if (!empty($esim['qr_code'])): ?>
        <div class="detail-card full-width">
            <div class="detail-card-title">
                <i class="fas fa-qrcode"></i>
                <span>QR Code</span>
            </div>
            
            <div class="qr-wrapper">
                <img src="<?php echo htmlspecialchars($esim['qr_code']); ?>" alt="QR Code">
                
                <?php if (!empty($esim['activation_code'])): ?>
                <a href="<?php echo htmlspecialchars($esim['activation_code']); ?>" 
                   class="btn-auto-add"
                   target="_blank">
                    <i class="fas fa-mobile-alt"></i>
                    <span>افزودن خودکار به گوشی</span>
                </a>
                <?php endif; ?>
            </div>
            
            <?php if (!empty($esim['activation_code'])): ?>
            <div class="activation-code-box">
                <label class="activation-label">کد فعال‌سازی:</label>
                <div class="activation-code" id="code-<?php echo $esim['id']; ?>">
                    <?php echo htmlspecialchars($esim['activation_code']); ?>
                </div>
                <button class="btn-copy" onclick="copyCode(<?php echo $esim['id']; ?>, '<?php echo htmlspecialchars($esim['activation_code']); ?>')">
                    <i class="fas fa-copy"></i>
                    کپی کد
                </button>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($esim['iccid'])): ?>
            <div class="activation-code-box">
                <label class="activation-label">ICCID:</label>
                <div class="activation-code">
                    <?php echo htmlspecialchars($esim['iccid']); ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>
        
        <!-- آموزش نصب -->
        <div class="detail-card full-width">
            <div class="detail-card-title">
                <i class="fas fa-graduation-cap"></i>
                <span>آموزش نصب eSIM</span>
            </div>
            
            <div class="instructions-content">
                <div class="instruction-section">
                    <h4><i class="fab fa-apple"></i> آیفون (iPhone):</h4>
                    <ol>
                        <li>به <strong>Settings</strong> بروید</li>
                        <li>روی <strong>Cellular</strong> یا <strong>Mobile Data</strong> ضربه بزنید</li>
                        <li><strong>Add Cellular Plan</strong> را انتخاب کنید</li>
                        <li>QR Code بالا را اسکن کنید</li>
                        <li>دستورالعمل‌های روی صفحه را دنبال کنید</li>
                    </ol>
                </div>
                
                <div class="instruction-section">
                    <h4><i class="fab fa-android"></i> اندروید (Android):</h4>
                    <ol>
                        <li>به <strong>Settings</strong> بروید</li>
                        <li><strong>Network & Internet</strong> را انتخاب کنید</li>
                        <li>روی <strong>Mobile Network</strong> ضربه بزنید</li>
                        <li><strong>Add</strong> یا <strong>+</strong> را بزنید</li>
                        <li>QR Code را اسکن کنید</li>
                    </ol>
                </div>
                
                <div class="note-box">
                    <i class="fas fa-info-circle"></i>
                    <div>
                        <strong>نکته مهم:</strong> قبل از نصب eSIM، اطمینان حاصل کنید که به اینترنت Wi-Fi متصل هستید.
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function copyCode(id, code) {
    navigator.clipboard.writeText(code).then(function() {
        alert('✅ کد با موفقیت کپی شد!');
    }, function() {
        alert('❌ خطا در کپی کردن کد');
    });
}
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>

